import os
from typing import List

from langchain_openai import ChatOpenAI, OpenAIEmbeddings
from langchain_core.messages import SystemMessage, HumanMessage, AIMessage

from langchain_community.document_loaders import PyPDFLoader
from langchain_text_splitters import RecursiveCharacterTextSplitter
from langchain_community.vectorstores import FAISS


# === 事前チェック：APIキー ===
if not os.getenv("OPENAI_API_KEY"):
    os.environ["OPENAI_API_KEY"] = (
        "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"
    )


# === 1. PDF を読み込んでベクトルストアを作成 ===
def build_vectorstore_from_pdf(pdf_path: str) -> FAISS:
    # PDF ローダー
    loader = PyPDFLoader(pdf_path)
    documents = loader.load()

    # テキスト分割（いい感じの長さに分割）
    text_splitter = RecursiveCharacterTextSplitter(
        chunk_size=1000,
        chunk_overlap=200,
    )
    splits = text_splitter.split_documents(documents)

    # 埋め込みモデル
    embeddings = OpenAIEmbeddings(model="text-embedding-3-small")

    # FAISS ベクトルストア作成
    vectorstore = FAISS.from_documents(splits, embedding=embeddings)
    return vectorstore


# === 2. PDF ベースで回答する関数 ===
def answer_with_rag(
    question: str,
    vectorstore: FAISS,
    llm: ChatOpenAI,
    chat_history: List,
):
    """質問と会話履歴を受け取り、PDFから関連部分を検索して RAG で回答（ストリーミング）"""

    # 類似ドキュメントを取得
    retriever = vectorstore.as_retriever(
        search_type="similarity",
        search_kwargs={"k": 4},
    )
    docs = retriever.invoke(question)

    # コンテキスト（PDF から取ってきたテキスト）をまとめる
    context_text = ""
    for i, d in enumerate(docs, start=1):
        context_text += f"[DOC{i}]\n{d.page_content}\n\n"

    # System メッセージ：RAG 用の指示
    system_prompt = (
        "あなたは PDF 資料に基づいて日本語で丁寧に答えるアシスタントです。\n"
        "与えられたコンテキストに書かれている内容を優先して回答してください。\n"
        "コンテキストに情報がない場合は、その旨を正直に伝えてください。\n"
    )

    # LLM へ渡すメッセージを組み立てる
    messages = [
        SystemMessage(content=system_prompt),
        SystemMessage(
            content=f"以下が PDF から取得したコンテキストです。\n\n{context_text}"
        ),
    ]

    # これまでのチャット履歴を追加
    messages.extend(chat_history)

    # 今回のユーザー質問
    messages.append(HumanMessage(content=question))

    # ストリーミングで出力
    print("AI  :", end="", flush=True)
    full_text = ""
    for chunk in llm.stream(messages):
        if chunk.content:
            full_text += chunk.content
            print(chunk.content, end="", flush=True)

    print("\n")
    return full_text


def main():
    # === 3. PDF のパスを指定 ===
    # 例: ./docs/manual.pdf など
    pdf_path = "白書とともに振り返る科学技術・イノベーション政策の歩み.pdf"  # ★ここを自分の PDF パスに変える

    if not os.path.exists(pdf_path):
        raise FileNotFoundError(f"PDF が見つかりません: {pdf_path}")

    print("PDF を読み込んでインデックスを作成中...")
    vectorstore = build_vectorstore_from_pdf(pdf_path)
    print("インデックス作成完了！ PDF に基づく Q&A ができます。\n")

    # ChatGPT モデル（ストリーミング有効）
    llm = ChatOpenAI(
        model="gpt-5.1",
        streaming=True,
    )

    # 会話履歴（HumanMessage / AIMessage だけ保持）
    chat_history: List = []

    print("=== PDF RAG チャット ===")
    print(f"対象PDF: {pdf_path}")
    print("PDF の内容について何でも聞いてください。")
    print("終了したいときは 'exit' または 'quit' と入力してください。\n")

    while True:
        user_input = input("あなた: ").strip()
        if user_input.lower() in {"exit", "quit"}:
            print("\nAI  : お疲れさまでした！また PDF の質問があればどうぞ。")
            break

        # RAG で回答（ストリーミング表示）
        answer = answer_with_rag(
            question=user_input,
            vectorstore=vectorstore,
            llm=llm,
            chat_history=chat_history,
        )

        # 履歴に追加（次回以降の文脈に使う）
        chat_history.append(HumanMessage(content=user_input))
        chat_history.append(AIMessage(content=answer))


if __name__ == "__main__":
    main()
